<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\FileStoreRequest;
use App\Http\Resources\FileResource;
use App\Http\Resources\FileResourceCollection;
use App\Models\File;
use App\Pipes\MinePipe;
use App\Pipes\SortPipe;
use Illuminate\Support\Facades\File as OsFile;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\Response;

class FileController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(File::class);
    }

    public static function routes(): void
    {
        Route::group(['prefix' => 'files'], function () {
            Route::get('', [self::class, 'index']);
            Route::post('', [self::class, 'store']);

            Route::delete('{file}', [self::class, 'destroy']);
        });
    }

    /**
     * @return object
     */
    public function index(): object
    {
        $files = File::pipe([
            MinePipe::class,
            SortPipe::class
        ])->page();

        return success(FileResourceCollection::make($files));
    }


    public function store(FileStoreRequest $request): object
    {
        $file = File::create($request->validated());

        return success(FileResource::make($file), status: Response::HTTP_CREATED);
    }

    public function destroy(File $file): object
    {
        if (in_array(Str::afterLast($file->path, '.'), ['gltf', 'fbx', 'obg', 'glb'])) {
            OsFile::deleteDirectory(public_path(Str::beforeLast($file->path, '/')));
        } else {
            OsFile::delete(public_path($file->path));
        }

        $file->delete();
        return success([], status: Response::HTTP_ACCEPTED);
    }
}
