<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\ProjectStoreRequest;
use App\Http\Resources\ProjectResource;
use App\Http\Resources\ProjectResourceCollection;
use App\Models\Project;
use App\Pipes\MinePipe;
use Illuminate\Support\Facades\Route;
use Symfony\Component\HttpFoundation\Response;

class ProjectController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Project::class);
    }

    /**
     * @return void
     */
    public static function routes(): void
    {
        Route::group(['prefix' => 'projects'], function () {
            Route::get('', [self::class, 'index']);
            Route::post('', [self::class, 'store']);

            Route::get('{project}', [self::class, 'show']);
            Route::put('{project}', [self::class, 'update']);

            Route::delete('{project}', [self::class, 'destroy']);
        });
    }

    /**
     * @return object
     */
    public function index(): object
    {
        $projects = Project::withCount('scenes')->pipe([
            MinePipe::class
        ])->page();

        return success(ProjectResourceCollection::make($projects));
    }

    /**
     * @param ProjectStoreRequest $request
     * @return object
     */
    public function store(ProjectStoreRequest $request): object
    {
        return success(
            ProjectResource::make(Project::create($request->validated())),
            status: Response::HTTP_CREATED
        );
    }

    /**
     * @param Project $project
     * @return object
     */
    public function show(Project $project): object
    {
        $project->load('scenes')->makeHidden('data');
        $project->loadCount('scenes');

        return success(ProjectResource::make($project));
    }

    /**
     * @param ProjectStoreRequest $request
     * @param Project $project
     * @return object
     */
    public function update(ProjectStoreRequest $request, Project $project): object
    {
        $project->update($request->validated());

        return success([], status: Response::HTTP_ACCEPTED);
    }

    /**
     * @param Project $project
     * @return object
     */
    public function destroy(Project $project): object
    {
        $project->delete();
        return success([], status: Response::HTTP_ACCEPTED);
    }
}
