<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\SceneStoreRequest;
use App\Http\Requests\SceneUpdateRequest;
use App\Http\Resources\SceneResource;
use App\Http\Resources\SceneResourceCollection;
use App\Models\Scene;
use App\Pipes\MinePipe;
use App\Pipes\SortPipe;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Symfony\Component\HttpFoundation\Response;

class SceneController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Scene::class);
    }

    /**
     * @return void
     */
    public static function routes(): void
    {
        Route::group(['prefix' => 'scenes'], function () {
            Route::get('', [self::class, 'index']);
            Route::post('', [self::class, 'store']);

            Route::get('{scene}', [self::class, 'show']);
            Route::put('{scene}', [self::class, 'update']);

            Route::delete('{scene}', [self::class, 'destroy']);
        });
    }

    public function index(): object
    {
        $scenes = Scene::pipe([
            MinePipe::class,
            SortPipe::class
        ])->page();

        return success(SceneResourceCollection::make($scenes));
    }

    /**
     * @param SceneStoreRequest $request
     * @return object
     */
    public function store(SceneStoreRequest $request): object
    {
        $scene = Scene::create(array_merge([
            Scene::USER_ID => Auth::id()
        ],$request->validated()));

        return success(SceneResource::make($scene), status: Response::HTTP_CREATED);
    }

    /**
     * @param Scene $scene
     * @return object
     */
    public function show(Scene $scene): object
    {
        return success(SceneResource::make($scene->append('data')));
    }

    public function update(SceneUpdateRequest $request, Scene $scene):object
    {
        $scene->update($request->validated());

        return success([], Response::HTTP_ACCEPTED);
    }

    public function destroy(Scene $scene)
    {
        $scene->delete();

        return success([], status:Response::HTTP_ACCEPTED);
    }
}
